<?php

namespace Database\Seeders;

use App\Models\Category;
use App\Models\Currency;
use App\Models\DeliveryZone;
use App\Models\Page;
use App\Models\PaymentMethod;
use App\Models\Product;
use App\Models\ProductWeight;
use App\Models\Setting;
use App\Models\User;
use App\Models\WeightUnit;
use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;

class DatabaseSeeder extends Seeder
{
    use WithoutModelEvents;

    /**
     * Seed the application's database.
     */
    public function run(): void
    {
        $this->seedAdmin();
        $weightUnits = $this->seedWeightUnits();
        $categories = $this->seedCategories();
        $this->seedProducts($categories, $weightUnits);
        $this->seedDeliveryZones();
        $this->seedPaymentMethods();
        $this->seedSettings();
        $this->seedCurrency();
        $this->seedPages();

        $this->command->info('✅ تم تهيئة قاعدة البيانات بنجاح - Database seeded successfully!');
    }

    // ──────────────────────────────────────────────────────────────
    //  1. Admin User
    // ──────────────────────────────────────────────────────────────

    private function seedAdmin(): void
    {
        User::updateOrCreate(
            ['email' => 'admin@rnahlah.com'],
            [
                'name'      => 'مدير النظام',
                'password'  => 'password',
                'role'      => 'admin',
                'phone'     => '0799999999',
                'is_active' => true,
            ]
        );

        $this->command->info('👤 Admin user created.');
    }

    // ──────────────────────────────────────────────────────────────
    //  2. Weight Units
    // ──────────────────────────────────────────────────────────────

    private function seedWeightUnits(): array
    {
        $units = [
            ['name_ar' => 'كيلو',      'name_en' => 'Kilogram',   'abbreviation_ar' => 'كغ',    'abbreviation_en' => 'kg'],
            ['name_ar' => 'جرام',      'name_en' => 'Gram',       'abbreviation_ar' => 'غ',     'abbreviation_en' => 'g'],
            ['name_ar' => 'لتر',       'name_en' => 'Liter',      'abbreviation_ar' => 'ل',     'abbreviation_en' => 'L'],
            ['name_ar' => 'مللي لتر',  'name_en' => 'Milliliter', 'abbreviation_ar' => 'مل',    'abbreviation_en' => 'ml'],
            ['name_ar' => 'حبة',       'name_en' => 'Piece',      'abbreviation_ar' => 'حبة',   'abbreviation_en' => 'pc'],
        ];

        $map = [];
        foreach ($units as $unit) {
            $record = WeightUnit::updateOrCreate(
                ['abbreviation_en' => $unit['abbreviation_en']],
                $unit
            );
            $map[$unit['abbreviation_en']] = $record;
        }

        $this->command->info('⚖️  Weight units created.');

        return $map;
    }

    // ──────────────────────────────────────────────────────────────
    //  3. Categories
    // ──────────────────────────────────────────────────────────────

    private function seedCategories(): array
    {
        $categories = [
            [
                'name_ar'        => 'عسل طبيعي',
                'name_en'        => 'Natural Honey',
                'slug'           => 'natural-honey',
                'description_ar' => 'تشكيلة واسعة من أجود أنواع العسل الطبيعي المستخلص من أزهار البرية الأردنية',
                'description_en' => 'A wide selection of the finest natural honey extracted from Jordanian wildflowers',
                'sort_order'     => 1,
                'is_active'      => true,
            ],
            [
                'name_ar'        => 'عسل مانوكا',
                'name_en'        => 'Manuka Honey',
                'slug'           => 'manuka-honey',
                'description_ar' => 'عسل مانوكا النيوزيلندي الأصلي بتركيزات مختلفة من الميثيل غليوكسال',
                'description_en' => 'Authentic New Zealand Manuka honey with various MGO concentrations',
                'sort_order'     => 2,
                'is_active'      => true,
            ],
            [
                'name_ar'        => 'منتجات النحل',
                'name_en'        => 'Bee Products',
                'slug'           => 'bee-products',
                'description_ar' => 'منتجات النحل الطبيعية من غذاء ملكات النحل وحبوب اللقاح والعكبر وشمع العسل',
                'description_en' => 'Natural bee products including royal jelly, pollen, propolis, and beeswax',
                'sort_order'     => 3,
                'is_active'      => true,
            ],
            [
                'name_ar'        => 'هدايا',
                'name_en'        => 'Gifts',
                'slug'           => 'gifts',
                'description_ar' => 'علب هدايا فاخرة تحتوي على مجموعة مختارة من أجود أنواع العسل ومنتجات النحل',
                'description_en' => 'Luxury gift boxes featuring a curated selection of premium honey and bee products',
                'sort_order'     => 4,
                'is_active'      => true,
            ],
            [
                'name_ar'        => 'عروض خاصة',
                'name_en'        => 'Special Offers',
                'slug'           => 'special-offers',
                'description_ar' => 'عروض وتخفيضات حصرية على منتجات مختارة من العسل ومنتجات النحل',
                'description_en' => 'Exclusive deals and discounts on selected honey and bee products',
                'sort_order'     => 5,
                'is_active'      => true,
            ],
        ];

        $map = [];
        foreach ($categories as $cat) {
            $record = Category::updateOrCreate(
                ['slug' => $cat['slug']],
                $cat
            );
            $map[$cat['slug']] = $record;
        }

        $this->command->info('📂 Categories created.');

        return $map;
    }

    // ──────────────────────────────────────────────────────────────
    //  4. Sample Products + Weights
    // ──────────────────────────────────────────────────────────────

    private function seedProducts(array $categories, array $weightUnits): void
    {
        $gram = $weightUnits['g'];
        $kg   = $weightUnits['kg'];

        $products = [
            // ── عسل السدر الطبيعي ──
            [
                'name_ar'              => 'عسل السدر الطبيعي',
                'name_en'              => 'Natural Sidr Honey',
                'slug'                 => 'natural-sidr-honey',
                'description_ar'       => 'عسل السدر الطبيعي من أجود أنواع العسل الأردني، يتميز بلونه الذهبي الداكن ونكهته الغنية. يُحصد من أشجار السدر البرية في المناطق الجبلية، ويُعتبر من أفضل أنواع العسل لفوائده الصحية المتعددة.',
                'description_en'       => 'Natural Sidr Honey is one of the finest Jordanian honeys, distinguished by its dark golden color and rich flavor. Harvested from wild Sidr trees in mountainous regions, it is considered one of the best types of honey for its numerous health benefits.',
                'short_description_ar' => 'عسل سدر طبيعي 100% من الجبال الأردنية',
                'short_description_en' => '100% natural Sidr honey from Jordanian mountains',
                'category_slug'        => 'natural-honey',
                'sku'                  => 'RN-SIDR-001',
                'base_price'           => 8.000,
                'is_featured'          => true,
                'is_new'               => false,
                'weights' => [
                    ['unit' => $gram, 'value' => 250,  'price' => 8.000,  'stock' => 50, 'sku' => 'RN-SIDR-250G',  'is_default' => true,  'sort_order' => 1],
                    ['unit' => $gram, 'value' => 500,  'price' => 15.000, 'stock' => 35, 'sku' => 'RN-SIDR-500G',  'is_default' => false, 'sort_order' => 2],
                    ['unit' => $kg,   'value' => 1,    'price' => 28.000, 'stock' => 20, 'sku' => 'RN-SIDR-1KG',   'is_default' => false, 'sort_order' => 3],
                ],
            ],

            // ── عسل الزهور البرية ──
            [
                'name_ar'              => 'عسل الزهور البرية',
                'name_en'              => 'Wild Flower Honey',
                'slug'                 => 'wild-flower-honey',
                'description_ar'       => 'عسل الزهور البرية الطبيعي يُجمع من رحيق الأزهار البرية المتنوعة في سهول الأردن وجباله. يتميز بطعمه اللطيف ولونه الذهبي الفاتح، وهو مثالي للاستخدام اليومي مع الأطعمة والمشروبات.',
                'description_en'       => 'Natural Wild Flower Honey is collected from the nectar of various wildflowers across Jordan\'s plains and mountains. It features a gentle taste and light golden color, perfect for daily use with foods and beverages.',
                'short_description_ar' => 'عسل زهور برية طبيعي متعدد الأزهار',
                'short_description_en' => 'Natural multi-floral wild flower honey',
                'category_slug'        => 'natural-honey',
                'sku'                  => 'RN-WFLR-001',
                'base_price'           => 6.000,
                'is_featured'          => true,
                'is_new'               => false,
                'weights' => [
                    ['unit' => $gram, 'value' => 250,  'price' => 6.000,  'stock' => 60, 'sku' => 'RN-WFLR-250G',  'is_default' => true,  'sort_order' => 1],
                    ['unit' => $gram, 'value' => 500,  'price' => 11.000, 'stock' => 40, 'sku' => 'RN-WFLR-500G',  'is_default' => false, 'sort_order' => 2],
                    ['unit' => $kg,   'value' => 1,    'price' => 20.000, 'stock' => 25, 'sku' => 'RN-WFLR-1KG',   'is_default' => false, 'sort_order' => 3],
                ],
            ],

            // ── عسل مانوكا MGO 400+ ──
            [
                'name_ar'              => 'عسل مانوكا MGO 400+',
                'name_en'              => 'Manuka Honey MGO 400+',
                'slug'                 => 'manuka-honey-mgo-400',
                'description_ar'       => 'عسل مانوكا النيوزيلندي الأصلي بتركيز MGO 400+. يُعد من أقوى أنواع العسل في خصائصه المضادة للبكتيريا. معتمد ومختبر لضمان الجودة والنقاء، مناسب للاستخدام العلاجي والعناية بالصحة.',
                'description_en'       => 'Authentic New Zealand Manuka Honey with MGO 400+ concentration. One of the most potent honeys for its antibacterial properties. Certified and tested for quality and purity, suitable for therapeutic use and health care.',
                'short_description_ar' => 'عسل مانوكا أصلي نيوزيلندي MGO 400+',
                'short_description_en' => 'Authentic New Zealand Manuka Honey MGO 400+',
                'category_slug'        => 'manuka-honey',
                'sku'                  => 'RN-MNK4-001',
                'base_price'           => 25.000,
                'is_featured'          => true,
                'is_new'               => true,
                'weights' => [
                    ['unit' => $gram, 'value' => 250,  'price' => 25.000, 'stock' => 20, 'sku' => 'RN-MNK4-250G',  'is_default' => true,  'sort_order' => 1],
                    ['unit' => $gram, 'value' => 500,  'price' => 45.000, 'stock' => 10, 'sku' => 'RN-MNK4-500G',  'is_default' => false, 'sort_order' => 2],
                ],
            ],

            // ── غذاء ملكات النحل ──
            [
                'name_ar'              => 'غذاء ملكات النحل',
                'name_en'              => 'Royal Jelly',
                'slug'                 => 'royal-jelly',
                'description_ar'       => 'غذاء ملكات النحل الطازج والنقي، يُعرف بفوائده الصحية الاستثنائية لتعزيز المناعة والطاقة. يُحفظ مبرداً للحفاظ على قيمته الغذائية العالية. منتج طبيعي 100% بدون أي إضافات.',
                'description_en'       => 'Fresh and pure Royal Jelly, known for its exceptional health benefits in boosting immunity and energy. Kept refrigerated to preserve its high nutritional value. 100% natural product with no additives.',
                'short_description_ar' => 'غذاء ملكات نحل طازج ونقي 100%',
                'short_description_en' => 'Fresh and pure 100% Royal Jelly',
                'category_slug'        => 'bee-products',
                'sku'                  => 'RN-RJLY-001',
                'base_price'           => 12.000,
                'is_featured'          => false,
                'is_new'               => true,
                'weights' => [
                    ['unit' => $gram, 'value' => 100,  'price' => 12.000, 'stock' => 30, 'sku' => 'RN-RJLY-100G',  'is_default' => true,  'sort_order' => 1],
                    ['unit' => $gram, 'value' => 250,  'price' => 25.000, 'stock' => 15, 'sku' => 'RN-RJLY-250G',  'is_default' => false, 'sort_order' => 2],
                ],
            ],

            // ── حبوب اللقاح ──
            [
                'name_ar'              => 'حبوب اللقاح',
                'name_en'              => 'Bee Pollen',
                'slug'                 => 'bee-pollen',
                'description_ar'       => 'حبوب لقاح النحل الطبيعية، غنية بالفيتامينات والمعادن والأحماض الأمينية. تُعتبر غذاءً متكاملاً يدعم صحة الجسم ويعزز الطاقة والحيوية. يمكن إضافتها إلى العصائر والسلطات والزبادي.',
                'description_en'       => 'Natural bee pollen, rich in vitamins, minerals, and amino acids. Considered a complete food that supports body health and boosts energy and vitality. Can be added to juices, salads, and yogurt.',
                'short_description_ar' => 'حبوب لقاح طبيعية غنية بالفيتامينات',
                'short_description_en' => 'Natural bee pollen rich in vitamins',
                'category_slug'        => 'bee-products',
                'sku'                  => 'RN-BPLN-001',
                'base_price'           => 7.000,
                'is_featured'          => false,
                'is_new'               => false,
                'weights' => [
                    ['unit' => $gram, 'value' => 200,  'price' => 7.000,  'stock' => 40, 'sku' => 'RN-BPLN-200G',  'is_default' => true,  'sort_order' => 1],
                    ['unit' => $gram, 'value' => 500,  'price' => 15.000, 'stock' => 20, 'sku' => 'RN-BPLN-500G',  'is_default' => false, 'sort_order' => 2],
                ],
            ],

            // ── شمع العسل الطبيعي ──
            [
                'name_ar'              => 'شمع العسل الطبيعي',
                'name_en'              => 'Natural Beeswax',
                'slug'                 => 'natural-beeswax',
                'description_ar'       => 'شمع عسل طبيعي نقي يُستخرج من خلايا النحل مباشرة. يُستخدم في العناية بالبشرة وصناعة الشموع الطبيعية والمراهم. غني بفيتامين A ومضادات الأكسدة، ويمكن مضغه للاستفادة من خصائصه العلاجية.',
                'description_en'       => 'Pure natural beeswax extracted directly from beehives. Used in skincare, natural candle making, and ointments. Rich in Vitamin A and antioxidants, it can be chewed to benefit from its therapeutic properties.',
                'short_description_ar' => 'شمع عسل طبيعي نقي من خلايا النحل',
                'short_description_en' => 'Pure natural beeswax from beehives',
                'category_slug'        => 'bee-products',
                'sku'                  => 'RN-BWAX-001',
                'base_price'           => 5.000,
                'is_featured'          => false,
                'is_new'               => false,
                'weights' => [
                    ['unit' => $gram, 'value' => 250,  'price' => 5.000,  'stock' => 45, 'sku' => 'RN-BWAX-250G',  'is_default' => true,  'sort_order' => 1],
                    ['unit' => $gram, 'value' => 500,  'price' => 9.000,  'stock' => 25, 'sku' => 'RN-BWAX-500G',  'is_default' => false, 'sort_order' => 2],
                ],
            ],
        ];

        foreach ($products as $data) {
            $weights = $data['weights'];
            $categorySlug = $data['category_slug'];
            unset($data['weights'], $data['category_slug']);

            $data['category_id'] = $categories[$categorySlug]->id;

            $product = Product::updateOrCreate(
                ['slug' => $data['slug']],
                $data
            );

            // Remove old weights and re-create
            $product->weights()->delete();

            foreach ($weights as $index => $w) {
                ProductWeight::create([
                    'product_id'     => $product->id,
                    'weight_unit_id' => $w['unit']->id,
                    'weight_value'   => $w['value'],
                    'price'          => $w['price'],
                    'stock'          => $w['stock'],
                    'sku'            => $w['sku'],
                    'is_default'     => $w['is_default'],
                    'sort_order'     => $w['sort_order'],
                ]);
            }
        }

        $this->command->info('🍯 Products & weights created.');
    }

    // ──────────────────────────────────────────────────────────────
    //  5. Delivery Zones
    // ──────────────────────────────────────────────────────────────

    private function seedDeliveryZones(): void
    {
        $zones = [
            [
                'name_ar'          => 'عمان',
                'name_en'          => 'Amman',
                'city'             => 'Amman',
                'delivery_fee'     => 2.000,
                'min_order_amount' => 5.000,
                'estimated_days'   => 1,
                'is_active'        => true,
                'sort_order'       => 1,
            ],
            [
                'name_ar'          => 'الزرقاء',
                'name_en'          => 'Zarqa',
                'city'             => 'Zarqa',
                'delivery_fee'     => 3.000,
                'min_order_amount' => 5.000,
                'estimated_days'   => 2,
                'is_active'        => true,
                'sort_order'       => 2,
            ],
            [
                'name_ar'          => 'إربد',
                'name_en'          => 'Irbid',
                'city'             => 'Irbid',
                'delivery_fee'     => 3.500,
                'min_order_amount' => 10.000,
                'estimated_days'   => 2,
                'is_active'        => true,
                'sort_order'       => 3,
            ],
            [
                'name_ar'          => 'باقي المناطق',
                'name_en'          => 'Other Areas',
                'city'             => 'Other',
                'delivery_fee'     => 4.000,
                'min_order_amount' => 10.000,
                'estimated_days'   => 3,
                'is_active'        => true,
                'sort_order'       => 4,
            ],
        ];

        foreach ($zones as $zone) {
            DeliveryZone::updateOrCreate(
                ['city' => $zone['city']],
                $zone
            );
        }

        $this->command->info('🚚 Delivery zones created.');
    }

    // ──────────────────────────────────────────────────────────────
    //  6. Payment Methods
    // ──────────────────────────────────────────────────────────────

    private function seedPaymentMethods(): void
    {
        $methods = [
            [
                'name_ar'          => 'الدفع عند الاستلام',
                'name_en'          => 'Cash on Delivery',
                'type'             => 'cash_on_delivery',
                'icon'             => 'cash',
                'instructions_ar'  => 'سيتم تحصيل المبلغ نقداً عند استلام الطلب. يرجى تجهيز المبلغ المطلوب.',
                'instructions_en'  => 'Payment will be collected in cash upon delivery. Please have the exact amount ready.',
                'fee'              => 0.000,
                'is_active'        => true,
                'sort_order'       => 1,
            ],
            [
                'name_ar'          => 'تحويل بنكي',
                'name_en'          => 'Bank Transfer',
                'type'             => 'bank_transfer',
                'icon'             => 'bank',
                'instructions_ar'  => 'يرجى تحويل المبلغ إلى حسابنا البنكي وإرسال إيصال التحويل عبر الواتساب. سيتم تأكيد الطلب بعد التحقق من التحويل.',
                'instructions_en'  => 'Please transfer the amount to our bank account and send the transfer receipt via WhatsApp. Order will be confirmed after verifying the transfer.',
                'fee'              => 0.000,
                'is_active'        => true,
                'sort_order'       => 2,
            ],
        ];

        foreach ($methods as $method) {
            PaymentMethod::updateOrCreate(
                ['type' => $method['type']],
                $method
            );
        }

        $this->command->info('💳 Payment methods created.');
    }

    // ──────────────────────────────────────────────────────────────
    //  7. Default Settings
    // ──────────────────────────────────────────────────────────────

    private function seedSettings(): void
    {
        $settings = [
            // ── General ──
            ['key' => 'store_name',        'value' => 'روابي نحلة',                                   'group' => 'general',  'type' => 'text'],
            ['key' => 'store_name_en',     'value' => 'Rawabi Nahlah',                                 'group' => 'general',  'type' => 'text'],
            ['key' => 'store_description', 'value' => 'متجر متخصص في بيع أجود أنواع العسل الطبيعي ومنتجات النحل في الأردن', 'group' => 'general', 'type' => 'text'],
            ['key' => 'store_description_en', 'value' => 'Specialized store selling the finest natural honey and bee products in Jordan', 'group' => 'general', 'type' => 'text'],
            ['key' => 'store_phone',       'value' => '0799999999',                                    'group' => 'general',  'type' => 'text'],
            ['key' => 'store_email',       'value' => 'info@rnahlah.com',                              'group' => 'general',  'type' => 'text'],
            ['key' => 'store_address',     'value' => 'عمان، الأردن',                                  'group' => 'general',  'type' => 'text'],
            ['key' => 'whatsapp_number',   'value' => '962799999999',                                  'group' => 'general',  'type' => 'text'],
            ['key' => 'currency',          'value' => 'JOD',                                           'group' => 'general',  'type' => 'text'],
            ['key' => 'currency_symbol',   'value' => 'د.ا',                                           'group' => 'general',  'type' => 'text'],

            // ── Social Media ──
            ['key' => 'social_facebook',   'value' => 'https://facebook.com/rawabinahlah',             'group' => 'social',   'type' => 'text'],
            ['key' => 'social_instagram',  'value' => 'https://instagram.com/rawabinahlah',            'group' => 'social',   'type' => 'text'],
            ['key' => 'social_tiktok',     'value' => 'https://tiktok.com/@rawabinahlah',              'group' => 'social',   'type' => 'text'],
            ['key' => 'social_twitter',    'value' => '',                                              'group' => 'social',   'type' => 'text'],

            // ── Features ──
            ['key' => 'splash_enabled',    'value' => 'true',                                          'group' => 'features', 'type' => 'boolean'],
            ['key' => 'reviews_enabled',   'value' => 'true',                                          'group' => 'features', 'type' => 'boolean'],
            ['key' => 'loyalty_enabled',   'value' => 'false',                                         'group' => 'features', 'type' => 'boolean'],
            ['key' => 'chat_enabled',      'value' => 'true',                                          'group' => 'features', 'type' => 'boolean'],
            ['key' => 'wishlist_enabled',  'value' => 'true',                                          'group' => 'features', 'type' => 'boolean'],
            ['key' => 'compare_enabled',   'value' => 'true',                                          'group' => 'features', 'type' => 'boolean'],
            ['key' => 'coupons_enabled',   'value' => 'true',                                          'group' => 'features', 'type' => 'boolean'],
            ['key' => 'reels_enabled',     'value' => 'false',                                         'group' => 'features', 'type' => 'boolean'],
            ['key' => 'stories_enabled',   'value' => 'false',                                         'group' => 'features', 'type' => 'boolean'],

            // ── Orders ──
            ['key' => 'min_order_amount',  'value' => '5',                                             'group' => 'orders',   'type' => 'number'],
            ['key' => 'free_delivery_min', 'value' => '30',                                            'group' => 'orders',   'type' => 'number'],
            ['key' => 'tax_percentage',    'value' => '0',                                             'group' => 'orders',   'type' => 'number'],

            // ── Appearance ──
            ['key' => 'primary_color',     'value' => '#D4A017',                                       'group' => 'appearance', 'type' => 'text'],
            ['key' => 'secondary_color',   'value' => '#8B6914',                                       'group' => 'appearance', 'type' => 'text'],
            ['key' => 'splash_duration',   'value' => '3',                                             'group' => 'appearance', 'type' => 'number'],
        ];

        foreach ($settings as $setting) {
            Setting::updateOrCreate(
                ['key' => $setting['key']],
                $setting
            );
        }

        $this->command->info('⚙️  Settings created.');
    }

    // ──────────────────────────────────────────────────────────────
    //  8. Default Currency
    // ──────────────────────────────────────────────────────────────

    private function seedCurrency(): void
    {
        Currency::updateOrCreate(
            ['code' => 'JOD'],
            [
                'name_ar'       => 'دينار أردني',
                'name_en'       => 'Jordanian Dinar',
                'symbol'        => 'د.ا',
                'exchange_rate'  => 1.0000,
                'is_active'     => true,
                'is_default'    => true,
            ]
        );

        $this->command->info('💰 Currency created.');
    }

    // ──────────────────────────────────────────────────────────────
    //  9. Sample Pages
    // ──────────────────────────────────────────────────────────────

    private function seedPages(): void
    {
        $pages = [
            [
                'title_ar'   => 'من نحن',
                'title_en'   => 'About Us',
                'slug'       => 'about-us',
                'content_ar' => '<h2>مرحباً بكم في روابي نحلة</h2>
<p>روابي نحلة هي علامة تجارية أردنية متخصصة في تقديم أجود أنواع العسل الطبيعي ومنتجات النحل. نحرص على اختيار منتجاتنا بعناية فائقة لنضمن لكم أعلى مستويات الجودة والنقاء.</p>
<h3>رسالتنا</h3>
<p>نسعى لتوفير العسل الطبيعي الأصيل ومنتجات النحل الصحية لكل بيت أردني، مع الحفاظ على أعلى معايير الجودة والأمانة في التعامل.</p>
<h3>رؤيتنا</h3>
<p>أن نكون الوجهة الأولى للعسل الطبيعي ومنتجات النحل في الأردن والمنطقة.</p>',
                'content_en' => '<h2>Welcome to Rawabi Nahlah</h2>
<p>Rawabi Nahlah is a Jordanian brand specializing in providing the finest natural honey and bee products. We carefully select our products to ensure the highest levels of quality and purity.</p>
<h3>Our Mission</h3>
<p>We strive to provide authentic natural honey and healthy bee products to every Jordanian home, while maintaining the highest standards of quality and integrity.</p>
<h3>Our Vision</h3>
<p>To be the premier destination for natural honey and bee products in Jordan and the region.</p>',
                'is_active'  => true,
                'sort_order'  => 1,
            ],
            [
                'title_ar'   => 'سياسة الخصوصية',
                'title_en'   => 'Privacy Policy',
                'slug'       => 'privacy-policy',
                'content_ar' => '<h2>سياسة الخصوصية</h2>
<p>نحن في روابي نحلة نقدّر خصوصيتكم ونلتزم بحماية بياناتكم الشخصية.</p>
<h3>جمع المعلومات</h3>
<p>نقوم بجمع المعلومات الضرورية فقط لإتمام عملية الشراء والتوصيل، بما في ذلك الاسم ورقم الهاتف والعنوان.</p>
<h3>استخدام المعلومات</h3>
<p>تُستخدم معلوماتكم فقط لمعالجة الطلبات والتوصيل والتواصل معكم بخصوص طلباتكم. لن نشارك بياناتكم مع أي طرف ثالث دون موافقتكم.</p>
<h3>حماية البيانات</h3>
<p>نتخذ جميع الإجراءات اللازمة لحماية بياناتكم من الوصول غير المصرح به أو الاستخدام غير القانوني.</p>',
                'content_en' => '<h2>Privacy Policy</h2>
<p>At Rawabi Nahlah, we value your privacy and are committed to protecting your personal data.</p>
<h3>Information Collection</h3>
<p>We collect only the necessary information to complete the purchase and delivery process, including name, phone number, and address.</p>
<h3>Information Usage</h3>
<p>Your information is used solely for order processing, delivery, and communication regarding your orders. We will not share your data with any third party without your consent.</p>
<h3>Data Protection</h3>
<p>We take all necessary measures to protect your data from unauthorized access or unlawful use.</p>',
                'is_active'  => true,
                'sort_order'  => 2,
            ],
            [
                'title_ar'   => 'شروط الاستخدام',
                'title_en'   => 'Terms & Conditions',
                'slug'       => 'terms-and-conditions',
                'content_ar' => '<h2>شروط الاستخدام</h2>
<p>باستخدامك لمتجر روابي نحلة فإنك توافق على الشروط والأحكام التالية:</p>
<h3>الطلبات والدفع</h3>
<p>جميع الأسعار المعروضة بالدينار الأردني وتشمل الضريبة. يتم تأكيد الطلب بعد إتمام عملية الدفع أو اختيار الدفع عند الاستلام.</p>
<h3>التوصيل</h3>
<p>نسعى لتوصيل الطلبات خلال المدة المحددة لكل منطقة. قد تتأخر بعض الطلبات بسبب ظروف خارجة عن إرادتنا.</p>
<h3>جودة المنتجات</h3>
<p>نضمن جودة جميع منتجاتنا. في حال وجود أي مشكلة في المنتج، يرجى التواصل معنا خلال 24 ساعة من الاستلام.</p>',
                'content_en' => '<h2>Terms & Conditions</h2>
<p>By using the Rawabi Nahlah store, you agree to the following terms and conditions:</p>
<h3>Orders and Payment</h3>
<p>All prices are displayed in Jordanian Dinar (JOD) and include tax. Orders are confirmed after completing the payment process or selecting cash on delivery.</p>
<h3>Delivery</h3>
<p>We strive to deliver orders within the specified time for each area. Some orders may be delayed due to circumstances beyond our control.</p>
<h3>Product Quality</h3>
<p>We guarantee the quality of all our products. If there is any issue with a product, please contact us within 24 hours of receiving it.</p>',
                'is_active'  => true,
                'sort_order'  => 3,
            ],
            [
                'title_ar'   => 'سياسة الاسترجاع',
                'title_en'   => 'Return Policy',
                'slug'       => 'return-policy',
                'content_ar' => '<h2>سياسة الاسترجاع والاستبدال</h2>
<p>نحرص في روابي نحلة على رضاكم التام عن منتجاتنا.</p>
<h3>شروط الاسترجاع</h3>
<ul>
<li>يمكن طلب الاسترجاع خلال 24 ساعة من استلام الطلب</li>
<li>يجب أن يكون المنتج بحالته الأصلية وغير مفتوح</li>
<li>يتحمل العميل تكلفة إعادة الشحن في حال عدم وجود عيب في المنتج</li>
</ul>
<h3>حالات الاسترجاع المقبولة</h3>
<ul>
<li>وصول منتج تالف أو مكسور</li>
<li>وصول منتج مختلف عن المطلوب</li>
<li>وجود عيب في جودة المنتج</li>
</ul>
<h3>طريقة الاسترجاع</h3>
<p>للمطالبة بالاسترجاع، يرجى التواصل معنا عبر الواتساب أو الاتصال بنا وسنقوم بمعالجة طلبكم في أسرع وقت ممكن.</p>',
                'content_en' => '<h2>Return & Exchange Policy</h2>
<p>At Rawabi Nahlah, we are committed to your complete satisfaction with our products.</p>
<h3>Return Conditions</h3>
<ul>
<li>Returns can be requested within 24 hours of receiving the order</li>
<li>The product must be in its original condition and unopened</li>
<li>The customer bears the return shipping cost if there is no defect in the product</li>
</ul>
<h3>Accepted Return Cases</h3>
<ul>
<li>Damaged or broken product upon arrival</li>
<li>Receiving a different product than ordered</li>
<li>Product quality defect</li>
</ul>
<h3>Return Process</h3>
<p>To request a return, please contact us via WhatsApp or call us, and we will process your request as soon as possible.</p>',
                'is_active'  => true,
                'sort_order'  => 4,
            ],
        ];

        foreach ($pages as $page) {
            Page::updateOrCreate(
                ['slug' => $page['slug']],
                $page
            );
        }

        $this->command->info('📄 Pages created.');
    }
}
