<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Category extends Model
{
    use HasFactory;

    /**
     * Accessors to include in array/JSON so API returns single "name" and "description" for frontend.
     */
    protected $appends = ['name', 'description'];

    protected $fillable = [
        'name_ar',
        'name_en',
        'slug',
        'description_ar',
        'description_en',
        'image',
        'parent_id',
        'sort_order',
        'is_active',
        'discount_percentage',
    ];

    protected function casts(): array
    {
        return [
            'is_active' => 'boolean',
            'discount_percentage' => 'decimal:2',
        ];
    }

    public function parent(): BelongsTo
    {
        return $this->belongsTo(Category::class, 'parent_id');
    }

    public function children(): HasMany
    {
        return $this->hasMany(Category::class, 'parent_id');
    }

    public function products(): HasMany
    {
        return $this->hasMany(Product::class);
    }

    /**
     * Get the display name (name_ar preferred, fallback to name_en).
     * Included in JSON so frontend receives a single "name" field.
     */
    protected function getNameAttribute(): string
    {
        return $this->attributes['name_ar'] ?? $this->attributes['name_en'] ?? '';
    }

    /**
     * Get the display description (description_ar preferred, fallback to description_en).
     * Included in JSON so frontend receives a single "description" field.
     */
    protected function getDescriptionAttribute(): string
    {
        return $this->attributes['description_ar'] ?? $this->attributes['description_en'] ?? '';
    }
}
