<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Wishlist;
use Illuminate\Http\Request;

class WishlistController extends Controller
{
    /**
     * Customer's wishlist.
     */
    public function index(Request $request)
    {
        $wishlists = Wishlist::where('customer_id', $request->user()->id)
            ->with(['product.images', 'product.weights.weightUnit', 'product.category'])
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json(['data' => $wishlists]);
    }

    /**
     * Add to wishlist (toggle behavior).
     */
    public function store(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
        ]);

        $customer = $request->user();

        $existing = Wishlist::where('customer_id', $customer->id)
            ->where('product_id', $request->product_id)
            ->first();

        if ($existing) {
            $existing->delete();
            return response()->json([
                'message' => 'تم إزالة المنتج من المفضلة',
                'wishlisted' => false,
            ]);
        }

        Wishlist::create([
            'customer_id' => $customer->id,
            'product_id' => $request->product_id,
        ]);

        return response()->json([
            'message' => 'تمت الإضافة إلى المفضلة',
            'wishlisted' => true,
        ], 201);
    }

    /**
     * Remove from wishlist.
     */
    public function destroy(Request $request, Product $product)
    {
        Wishlist::where('customer_id', $request->user()->id)
            ->where('product_id', $product->id)
            ->delete();

        return response()->json([
            'message' => 'تم إزالة المنتج من المفضلة',
        ]);
    }
}
