<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    /**
     * Paginated products with filters.
     */
    public function index(Request $request)
    {
        $query = Product::with(['weights.weightUnit', 'images', 'category'])
            ->where('is_active', true);

        // Filter by category
        if ($request->filled('category')) {
            $query->where('category_id', $request->category);
        }

        // Filter by price range
        if ($request->filled('price_min')) {
            $query->where('base_price', '>=', $request->price_min);
        }

        if ($request->filled('price_max')) {
            $query->where('base_price', '<=', $request->price_max);
        }

        // Filter featured
        if ($request->filled('is_featured')) {
            $query->where('is_featured', $request->boolean('is_featured'));
        }

        // Filter new
        if ($request->filled('is_new')) {
            $query->where('is_new', $request->boolean('is_new'));
        }

        // Sort
        $sortField = $request->get('sort', 'created_at');
        $sortDirection = $request->get('direction', 'desc');

        $allowedSorts = ['created_at', 'base_price', 'name_ar', 'name_en', 'sales_count', 'avg_rating', 'views_count'];
        if (in_array($sortField, $allowedSorts)) {
            $query->orderBy($sortField, $sortDirection === 'asc' ? 'asc' : 'desc');
        }

        $perPage = min($request->get('per_page', 15), 50);

        return response()->json($query->paginate($perPage));
    }

    /**
     * Featured products.
     */
    public function featured()
    {
        $products = Product::with(['weights.weightUnit', 'images', 'category'])
            ->where('is_active', true)
            ->where('is_featured', true)
            ->orderBy('created_at', 'desc')
            ->limit(20)
            ->get();

        return response()->json(['data' => $products]);
    }

    /**
     * Best-selling products.
     */
    public function bestSellers()
    {
        $products = Product::with(['weights.weightUnit', 'images', 'category'])
            ->where('is_active', true)
            ->where('sales_count', '>', 0)
            ->orderBy('sales_count', 'desc')
            ->limit(20)
            ->get();

        return response()->json(['data' => $products]);
    }

    /**
     * Search products by name or description.
     */
    public function search(Request $request)
    {
        $request->validate([
            'q' => 'required|string|min:2|max:100',
        ]);

        $search = $request->q;

        $products = Product::with(['weights.weightUnit', 'images', 'category'])
            ->where('is_active', true)
            ->where(function ($query) use ($search) {
                $query->where('name_ar', 'like', "%{$search}%")
                    ->orWhere('name_en', 'like', "%{$search}%")
                    ->orWhere('description_ar', 'like', "%{$search}%")
                    ->orWhere('description_en', 'like', "%{$search}%")
                    ->orWhere('sku', 'like', "%{$search}%");
            })
            ->orderBy('sales_count', 'desc')
            ->paginate(15);

        return response()->json($products);
    }

    /**
     * Single product with all relations.
     */
    public function show(Product $product)
    {
        if (!$product->is_active) {
            return response()->json(['message' => 'المنتج غير متوفر'], 404);
        }

        $product->load(['weights.weightUnit', 'images', 'videos', 'category', 'reviews' => function ($q) {
            $q->where('status', 'approved')->latest()->limit(10);
        }]);

        // Increment views
        $product->increment('views_count');

        return response()->json(['data' => $product]);
    }
}
