<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Cart;
use App\Models\Coupon;
use App\Models\Order;
use App\Models\OrderItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class OrderController extends Controller
{
    /**
     * Customer's orders, paginated.
     */
    public function index(Request $request)
    {
        $orders = Order::where('customer_id', $request->user()->id)
            ->with(['items.product.images', 'items.productWeight.weightUnit'])
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 10));

        return response()->json($orders);
    }

    /**
     * Create order from cart.
     */
    public function store(Request $request)
    {
        $request->validate([
            'delivery_address' => 'required|string|max:500',
            'delivery_city' => 'required|string|max:100',
            'delivery_zone_id' => 'nullable|exists:delivery_zones,id',
            'payment_method_id' => 'nullable|exists:payment_methods,id',
            'coupon_code' => 'nullable|string|max:50',
            'notes' => 'nullable|string|max:1000',
            'phone' => 'nullable|string|max:20',
        ]);

        $customer = $request->user();

        $cart = Cart::with(['items.product', 'items.productWeight'])
            ->where('customer_id', $customer->id)
            ->first();

        if (!$cart || $cart->items->isEmpty()) {
            return response()->json(['message' => 'السلة فارغة'], 422);
        }

        return DB::transaction(function () use ($request, $customer, $cart) {
            // Calculate subtotal
            $subtotal = 0;
            foreach ($cart->items as $item) {
                if (!$item->product->is_active) {
                    return response()->json([
                        'message' => "المنتج {$item->product->name_ar} غير متوفر حالياً",
                    ], 422);
                }

                if ($item->productWeight->stock < $item->quantity) {
                    return response()->json([
                        'message' => "الكمية المطلوبة من {$item->product->name_ar} غير متوفرة",
                    ], 422);
                }

                $subtotal += $item->productWeight->price * $item->quantity;
            }

            // Apply coupon
            $discount = 0;
            $couponId = null;
            if ($request->filled('coupon_code')) {
                $coupon = Coupon::where('code', $request->coupon_code)
                    ->where('is_active', true)
                    ->where(function ($q) {
                        $q->whereNull('start_date')->orWhere('start_date', '<=', now());
                    })
                    ->where(function ($q) {
                        $q->whereNull('end_date')->orWhere('end_date', '>=', now());
                    })
                    ->first();

                if ($coupon) {
                    if ($coupon->min_order_amount && $subtotal < $coupon->min_order_amount) {
                        return response()->json([
                            'message' => "الحد الأدنى للطلب لاستخدام الكوبون هو {$coupon->min_order_amount}",
                        ], 422);
                    }

                    if ($coupon->max_uses && $coupon->used_count >= $coupon->max_uses) {
                        return response()->json(['message' => 'تم استنفاد الكوبون'], 422);
                    }

                    $discount = $coupon->type === 'percentage'
                        ? $subtotal * ($coupon->value / 100)
                        : $coupon->value;

                    if ($coupon->max_discount && $discount > $coupon->max_discount) {
                        $discount = $coupon->max_discount;
                    }

                    $couponId = $coupon->id;
                    $coupon->increment('used_count');
                }
            }

            // Delivery fee (can be calculated from zone)
            $deliveryFee = 0;
            if ($request->filled('delivery_zone_id')) {
                $zone = \App\Models\DeliveryZone::find($request->delivery_zone_id);
                $deliveryFee = $zone ? $zone->delivery_fee : 0;
            }

            $total = $subtotal - $discount + $deliveryFee;

            // Generate order number: RN-YYYYMMDD-XXXX
            $today = now()->format('Ymd');
            $todayCount = Order::whereDate('created_at', now()->toDateString())->count() + 1;
            $orderNumber = 'RN-' . $today . '-' . str_pad($todayCount, 4, '0', STR_PAD_LEFT);

            // Create order
            $order = Order::create([
                'order_number' => $orderNumber,
                'customer_id' => $customer->id,
                'coupon_id' => $couponId,
                'delivery_zone_id' => $request->delivery_zone_id,
                'payment_method_id' => $request->payment_method_id,
                'subtotal' => round($subtotal, 3),
                'discount_amount' => round($discount, 3),
                'delivery_fee' => round($deliveryFee, 3),
                'total' => round($total, 3),
                'status' => 'pending',
                'delivery_address' => $request->delivery_address,
                'delivery_city' => $request->delivery_city,
                'notes' => $request->notes,
            ]);

            // Create order items
            foreach ($cart->items as $cartItem) {
                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $cartItem->product_id,
                    'product_weight_id' => $cartItem->product_weight_id,
                    'product_name' => $cartItem->product->name_ar,
                    'weight_label' => $cartItem->productWeight->weight_value . ' ' .
                        ($cartItem->productWeight->weightUnit->abbreviation_ar ?? ''),
                    'unit_price' => $cartItem->productWeight->price,
                    'quantity' => $cartItem->quantity,
                    'total_price' => $cartItem->productWeight->price * $cartItem->quantity,
                    'notes' => $cartItem->notes,
                ]);

                // Decrement stock
                $cartItem->productWeight->decrement('stock', $cartItem->quantity);
                // Increment sales count
                $cartItem->product->increment('sales_count', $cartItem->quantity);
            }

            // Update customer stats
            $customer->increment('total_orders');
            $customer->increment('total_spent', round($total, 3));
            $customer->update(['last_order_at' => now()]);

            // Clear cart
            $cart->items()->delete();
            $cart->update(['coupon_id' => null]);

            // Load relations
            $order->load(['items.product.images', 'items.productWeight.weightUnit']);

            // Build WhatsApp message
            $whatsappMsg = $this->buildWhatsAppMessage($order, $customer);

            return response()->json([
                'message' => 'تم إنشاء الطلب بنجاح',
                'data' => $order,
                'whatsapp_message' => $whatsappMsg,
            ], 201);
        });
    }

    /**
     * Single order detail.
     */
    public function show(Request $request, Order $order)
    {
        if ($order->customer_id !== $request->user()->id) {
            return response()->json(['message' => 'غير مصرح'], 403);
        }

        $order->load([
            'items.product.images',
            'items.productWeight.weightUnit',
            'statuses',
            'coupon',
            'deliveryZone',
            'paymentMethod',
        ]);

        return response()->json(['data' => $order]);
    }

    /**
     * Public order tracking by order number.
     */
    public function track(string $orderNumber)
    {
        $order = Order::where('order_number', $orderNumber)
            ->with(['statuses' => function ($q) {
                $q->orderBy('created_at', 'asc');
            }])
            ->first();

        if (!$order) {
            return response()->json(['message' => 'رقم الطلب غير موجود'], 404);
        }

        return response()->json([
            'data' => [
                'order_number' => $order->order_number,
                'status' => $order->status,
                'total' => $order->total,
                'created_at' => $order->created_at,
                'timeline' => $order->statuses->map(function ($status) {
                    return [
                        'status' => $status->status,
                        'notes' => $status->notes,
                        'changed_at' => $status->created_at,
                    ];
                }),
            ],
        ]);
    }

    /**
     * Build WhatsApp order summary message.
     */
    private function buildWhatsAppMessage(Order $order, $customer): string
    {
        $lines = [];
        $lines[] = "🛒 طلب جديد #{$order->order_number}";
        $lines[] = "👤 {$customer->name} - {$customer->phone}";
        $lines[] = "📍 {$order->delivery_address}, {$order->delivery_city}";
        $lines[] = "---";

        foreach ($order->items as $item) {
            $lines[] = "• {$item->product_name} ({$item->weight_label}) x{$item->quantity} = {$item->total}";
        }

        $lines[] = "---";
        $lines[] = "المجموع الفرعي: {$order->subtotal}";
        if ($order->discount_amount > 0) {
            $lines[] = "الخصم: -{$order->discount_amount}";
        }
        if ($order->delivery_fee > 0) {
            $lines[] = "التوصيل: {$order->delivery_fee}";
        }
        $lines[] = "الإجمالي: {$order->total}";

        if ($order->notes) {
            $lines[] = "ملاحظات: {$order->notes}";
        }

        return implode("\n", $lines);
    }
}
