<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Comparison;
use App\Models\Product;
use Illuminate\Http\Request;

class ComparisonController extends Controller
{
    /**
     * Customer's comparison list.
     */
    public function index(Request $request)
    {
        $comparisons = Comparison::where('customer_id', $request->user()->id)
            ->with(['product.images', 'product.weights.weightUnit', 'product.category'])
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json(['data' => $comparisons]);
    }

    /**
     * Add product to comparison.
     */
    public function store(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
        ]);

        $customer = $request->user();

        // Limit comparison items
        $count = Comparison::where('customer_id', $customer->id)->count();
        if ($count >= 4) {
            return response()->json([
                'message' => 'يمكنك مقارنة 4 منتجات كحد أقصى',
            ], 422);
        }

        $existing = Comparison::where('customer_id', $customer->id)
            ->where('product_id', $request->product_id)
            ->first();

        if ($existing) {
            return response()->json([
                'message' => 'المنتج موجود بالفعل في قائمة المقارنة',
            ], 422);
        }

        $comparison = Comparison::create([
            'customer_id' => $customer->id,
            'product_id' => $request->product_id,
        ]);

        $comparison->load(['product.images', 'product.weights.weightUnit']);

        return response()->json([
            'message' => 'تمت الإضافة إلى المقارنة',
            'data' => $comparison,
        ], 201);
    }

    /**
     * Remove product from comparison.
     */
    public function destroy(Request $request, Product $product)
    {
        Comparison::where('customer_id', $request->user()->id)
            ->where('product_id', $product->id)
            ->delete();

        return response()->json([
            'message' => 'تم إزالة المنتج من المقارنة',
        ]);
    }
}
