<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\Story;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class StoryController extends Controller
{
    /**
     * List all stories.
     */
    public function index()
    {
        $stories = Story::orderBy('sort_order', 'asc')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json(['data' => $stories]);
    }

    /**
     * Create story.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title_ar' => 'nullable|string|max:255',
            'title_en' => 'nullable|string|max:255',
            'media' => 'required|file|max:20480',
            'media_type' => 'required|in:image,video',
            'thumbnail' => 'nullable|image|max:2048',
            'link' => 'nullable|string|max:500',
            'sort_order' => 'integer|min:0',
            'is_active' => 'boolean',
            'expires_at' => 'nullable|date',
        ]);

        $mediaPath = $request->file('media')->store('stories', 'public');
        $thumbnailPath = null;
        if ($request->hasFile('thumbnail')) {
            $thumbnailPath = $request->file('thumbnail')->store('stories/thumbnails', 'public');
        }

        $story = Story::create([
            'title_ar' => $request->title_ar,
            'title_en' => $request->title_en,
            'media' => $mediaPath,
            'media_type' => $request->media_type,
            'thumbnail' => $thumbnailPath,
            'link' => $request->link,
            'sort_order' => $request->get('sort_order', 0),
            'is_active' => $request->boolean('is_active', true),
            'expires_at' => $request->expires_at,
        ]);

        return response()->json([
            'message' => 'تم إنشاء الستوري بنجاح',
            'data' => $story,
        ], 201);
    }

    /**
     * Show single story.
     */
    public function show(Story $story)
    {
        return response()->json(['data' => $story]);
    }

    /**
     * Update story.
     */
    public function update(Request $request, Story $story)
    {
        $request->validate([
            'title_ar' => 'nullable|string|max:255',
            'title_en' => 'nullable|string|max:255',
            'media' => 'nullable|file|max:20480',
            'media_type' => 'nullable|in:image,video',
            'thumbnail' => 'nullable|image|max:2048',
            'link' => 'nullable|string|max:500',
            'sort_order' => 'integer|min:0',
            'is_active' => 'boolean',
            'expires_at' => 'nullable|date',
        ]);

        if ($request->hasFile('media')) {
            Storage::disk('public')->delete($story->media);
            $story->media = $request->file('media')->store('stories', 'public');
        }

        if ($request->hasFile('thumbnail')) {
            if ($story->thumbnail) {
                Storage::disk('public')->delete($story->thumbnail);
            }
            $story->thumbnail = $request->file('thumbnail')->store('stories/thumbnails', 'public');
        }

        $story->update(array_merge(
            $request->except(['media', 'thumbnail']),
            ['media' => $story->media, 'thumbnail' => $story->thumbnail]
        ));

        return response()->json([
            'message' => 'تم تحديث الستوري بنجاح',
            'data' => $story,
        ]);
    }

    /**
     * Delete story.
     */
    public function destroy(Story $story)
    {
        Storage::disk('public')->delete($story->media);
        if ($story->thumbnail) {
            Storage::disk('public')->delete($story->thumbnail);
        }
        $story->delete();

        return response()->json([
            'message' => 'تم حذف الستوري بنجاح',
        ]);
    }
}
