<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class SettingController extends Controller
{
    /**
     * List all settings.
     */
    public function index()
    {
        $settings = Setting::orderBy('key', 'asc')
            ->get()
            ->groupBy('group');

        return response()->json(['data' => $settings]);
    }

    /**
     * Update a setting.
     */
    public function update(Request $request)
    {
        $request->validate([
            'settings' => 'required|array',
            'settings.*.key' => 'required|string|max:100',
            'settings.*.value' => 'nullable|string',
        ]);

        foreach ($request->settings as $item) {
            Setting::updateOrCreate(
                ['key' => $item['key']],
                [
                    'value' => $item['value'] ?? '',
                    'group' => $item['group'] ?? 'general',
                    'is_public' => $item['is_public'] ?? true,
                ]
            );
        }

        return response()->json([
            'message' => 'تم تحديث الإعدادات بنجاح',
        ]);
    }

    /**
     * Upload logo.
     */
    public function uploadLogo(Request $request)
    {
        $request->validate([
            'logo' => 'required|image|max:2048',
        ]);

        // Delete old logo
        $oldLogo = Setting::where('key', 'site_logo')->first();
        if ($oldLogo && $oldLogo->value) {
            Storage::disk('public')->delete($oldLogo->value);
        }

        $path = $request->file('logo')->store('settings', 'public');

        Setting::updateOrCreate(
            ['key' => 'site_logo'],
            ['value' => $path, 'group' => 'branding', 'is_public' => true]
        );

        return response()->json([
            'message' => 'تم تحديث الشعار بنجاح',
            'data' => ['path' => $path],
        ]);
    }

    /**
     * Upload favicon.
     */
    public function uploadFavicon(Request $request)
    {
        $request->validate([
            'favicon' => 'required|image|max:512',
        ]);

        // Delete old favicon
        $oldFavicon = Setting::where('key', 'site_favicon')->first();
        if ($oldFavicon && $oldFavicon->value) {
            Storage::disk('public')->delete($oldFavicon->value);
        }

        $path = $request->file('favicon')->store('settings', 'public');

        Setting::updateOrCreate(
            ['key' => 'site_favicon'],
            ['value' => $path, 'group' => 'branding', 'is_public' => true]
        );

        return response()->json([
            'message' => 'تم تحديث الأيقونة بنجاح',
            'data' => ['path' => $path],
        ]);
    }

    /**
     * Upload hero background image for homepage.
     */
    public function uploadHeroBackground(Request $request)
    {
        $request->validate([
            'hero_background' => 'required|image|max:5120',
        ]);

        $oldBg = Setting::where('key', 'hero_background_image')->first();
        if ($oldBg && $oldBg->value) {
            Storage::disk('public')->delete($oldBg->value);
        }

        $path = $request->file('hero_background')->store('settings', 'public');

        Setting::updateOrCreate(
            ['key' => 'hero_background_image'],
            ['value' => $path, 'group' => 'appearance']
        );

        return response()->json([
            'message' => 'تم تحديث صورة الخلفية بنجاح',
            'data' => ['path' => $path],
        ]);
    }
}
