<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\Seo;
use Illuminate\Http\Request;

class SeoController extends Controller
{
    /**
     * List all SEO entries.
     */
    public function index(Request $request)
    {
        $query = Seo::query();

        if ($request->filled('type')) {
            $query->where('seoable_type', $request->type);
        }

        return response()->json(
            $query->orderBy('created_at', 'desc')
                ->paginate($request->get('per_page', 20))
        );
    }

    /**
     * Create SEO entry.
     */
    public function store(Request $request)
    {
        $request->validate([
            'seoable_type' => 'required|string|max:255',
            'seoable_id' => 'required|integer',
            'meta_title_ar' => 'nullable|string|max:255',
            'meta_title_en' => 'nullable|string|max:255',
            'meta_description_ar' => 'nullable|string|max:500',
            'meta_description_en' => 'nullable|string|max:500',
            'meta_keywords_ar' => 'nullable|string|max:500',
            'meta_keywords_en' => 'nullable|string|max:500',
            'og_title' => 'nullable|string|max:255',
            'og_description' => 'nullable|string|max:500',
            'og_image' => 'nullable|string|max:500',
            'canonical_url' => 'nullable|string|max:500',
        ]);

        $seo = Seo::create($request->all());

        return response()->json([
            'message' => 'تم إنشاء بيانات SEO بنجاح',
            'data' => $seo,
        ], 201);
    }

    /**
     * Show single SEO entry.
     */
    public function show(Seo $seo)
    {
        return response()->json(['data' => $seo]);
    }

    /**
     * Update SEO entry.
     */
    public function update(Request $request, Seo $seo)
    {
        $request->validate([
            'meta_title_ar' => 'nullable|string|max:255',
            'meta_title_en' => 'nullable|string|max:255',
            'meta_description_ar' => 'nullable|string|max:500',
            'meta_description_en' => 'nullable|string|max:500',
            'meta_keywords_ar' => 'nullable|string|max:500',
            'meta_keywords_en' => 'nullable|string|max:500',
            'og_title' => 'nullable|string|max:255',
            'og_description' => 'nullable|string|max:500',
            'og_image' => 'nullable|string|max:500',
            'canonical_url' => 'nullable|string|max:500',
        ]);

        $seo->update($request->all());

        return response()->json([
            'message' => 'تم تحديث بيانات SEO بنجاح',
            'data' => $seo,
        ]);
    }

    /**
     * Delete SEO entry.
     */
    public function destroy(Seo $seo)
    {
        $seo->delete();

        return response()->json([
            'message' => 'تم حذف بيانات SEO بنجاح',
        ]);
    }
}
