<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\Reel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ReelController extends Controller
{
    /**
     * List all reels.
     */
    public function index()
    {
        $reels = Reel::with(['product:id,name_ar,name_en,slug'])
            ->orderBy('sort_order', 'asc')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json(['data' => $reels]);
    }

    /**
     * Create reel.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title_ar' => 'nullable|string|max:255',
            'title_en' => 'nullable|string|max:255',
            'video' => 'required|file|mimes:mp4,mov|max:51200',
            'thumbnail' => 'nullable|image|max:2048',
            'product_id' => 'nullable|exists:products,id',
            'sort_order' => 'integer|min:0',
            'is_active' => 'boolean',
        ]);

        $videoPath = $request->file('video')->store('reels', 'public');
        $thumbnailPath = null;
        if ($request->hasFile('thumbnail')) {
            $thumbnailPath = $request->file('thumbnail')->store('reels/thumbnails', 'public');
        }

        $reel = Reel::create([
            'title_ar' => $request->title_ar,
            'title_en' => $request->title_en,
            'video' => $videoPath,
            'thumbnail' => $thumbnailPath,
            'product_id' => $request->product_id,
            'sort_order' => $request->get('sort_order', 0),
            'is_active' => $request->boolean('is_active', true),
        ]);

        return response()->json([
            'message' => 'تم إنشاء الريل بنجاح',
            'data' => $reel,
        ], 201);
    }

    /**
     * Show single reel.
     */
    public function show(Reel $reel)
    {
        $reel->load(['product:id,name_ar,name_en,slug']);

        return response()->json(['data' => $reel]);
    }

    /**
     * Update reel.
     */
    public function update(Request $request, Reel $reel)
    {
        $request->validate([
            'title_ar' => 'nullable|string|max:255',
            'title_en' => 'nullable|string|max:255',
            'video' => 'nullable|file|mimes:mp4,mov|max:51200',
            'thumbnail' => 'nullable|image|max:2048',
            'product_id' => 'nullable|exists:products,id',
            'sort_order' => 'integer|min:0',
            'is_active' => 'boolean',
        ]);

        if ($request->hasFile('video')) {
            Storage::disk('public')->delete($reel->video);
            $reel->video = $request->file('video')->store('reels', 'public');
        }

        if ($request->hasFile('thumbnail')) {
            if ($reel->thumbnail) {
                Storage::disk('public')->delete($reel->thumbnail);
            }
            $reel->thumbnail = $request->file('thumbnail')->store('reels/thumbnails', 'public');
        }

        $reel->update(array_merge(
            $request->except(['video', 'thumbnail']),
            ['video' => $reel->video, 'thumbnail' => $reel->thumbnail]
        ));

        return response()->json([
            'message' => 'تم تحديث الريل بنجاح',
            'data' => $reel,
        ]);
    }

    /**
     * Delete reel.
     */
    public function destroy(Reel $reel)
    {
        Storage::disk('public')->delete($reel->video);
        if ($reel->thumbnail) {
            Storage::disk('public')->delete($reel->thumbnail);
        }
        $reel->delete();

        return response()->json([
            'message' => 'تم حذف الريل بنجاح',
        ]);
    }
}
