<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\Popup;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class PopupController extends Controller
{
    /**
     * List all popups.
     */
    public function index()
    {
        $popups = Popup::orderBy('created_at', 'desc')->get();

        return response()->json(['data' => $popups]);
    }

    /**
     * Create popup.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title_ar' => 'nullable|string|max:255',
            'title_en' => 'nullable|string|max:255',
            'content_ar' => 'nullable|string',
            'content_en' => 'nullable|string',
            'image' => 'nullable|image|max:5120',
            'link' => 'nullable|string|max:500',
            'button_text_ar' => 'nullable|string|max:100',
            'button_text_en' => 'nullable|string|max:100',
            'is_active' => 'boolean',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'show_once' => 'boolean',
            'delay_seconds' => 'nullable|integer|min:0',
        ]);

        $imagePath = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('popups', 'public');
        }

        $popup = Popup::create(array_merge(
            $request->except('image'),
            [
                'image' => $imagePath,
                'is_active' => $request->boolean('is_active', true),
                'show_once' => $request->boolean('show_once', false),
            ]
        ));

        return response()->json([
            'message' => 'تم إنشاء النافذة المنبثقة بنجاح',
            'data' => $popup,
        ], 201);
    }

    /**
     * Show single popup.
     */
    public function show(Popup $popup)
    {
        return response()->json(['data' => $popup]);
    }

    /**
     * Update popup.
     */
    public function update(Request $request, Popup $popup)
    {
        $request->validate([
            'title_ar' => 'nullable|string|max:255',
            'title_en' => 'nullable|string|max:255',
            'content_ar' => 'nullable|string',
            'content_en' => 'nullable|string',
            'image' => 'nullable|image|max:5120',
            'link' => 'nullable|string|max:500',
            'button_text_ar' => 'nullable|string|max:100',
            'button_text_en' => 'nullable|string|max:100',
            'is_active' => 'boolean',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'show_once' => 'boolean',
            'delay_seconds' => 'nullable|integer|min:0',
        ]);

        if ($request->hasFile('image')) {
            if ($popup->image) {
                Storage::disk('public')->delete($popup->image);
            }
            $popup->image = $request->file('image')->store('popups', 'public');
        }

        $popup->update(array_merge(
            $request->except('image'),
            ['image' => $popup->image]
        ));

        return response()->json([
            'message' => 'تم تحديث النافذة المنبثقة بنجاح',
            'data' => $popup,
        ]);
    }

    /**
     * Delete popup.
     */
    public function destroy(Popup $popup)
    {
        if ($popup->image) {
            Storage::disk('public')->delete($popup->image);
        }
        $popup->delete();

        return response()->json([
            'message' => 'تم حذف النافذة المنبثقة بنجاح',
        ]);
    }
}
