<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\PaymentMethod;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class PaymentMethodController extends Controller
{
    /**
     * List all payment methods.
     */
    public function index()
    {
        $methods = PaymentMethod::orderBy('sort_order', 'asc')->get();

        return response()->json(['data' => $methods]);
    }

    /**
     * Create payment method.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name_ar' => 'required|string|max:255',
            'name_en' => 'required|string|max:255',
            'description_ar' => 'nullable|string|max:500',
            'description_en' => 'nullable|string|max:500',
            'icon' => 'nullable|image|max:1024',
            'type' => 'required|string|max:50',
            'instructions_ar' => 'nullable|string',
            'instructions_en' => 'nullable|string',
            'sort_order' => 'integer|min:0',
            'is_active' => 'boolean',
        ]);

        $iconPath = null;
        if ($request->hasFile('icon')) {
            $iconPath = $request->file('icon')->store('payment-methods', 'public');
        }

        $method = PaymentMethod::create(array_merge(
            $request->except('icon'),
            [
                'icon' => $iconPath,
                'is_active' => $request->boolean('is_active', true),
                'sort_order' => $request->get('sort_order', 0),
            ]
        ));

        return response()->json([
            'message' => 'تم إنشاء طريقة الدفع بنجاح',
            'data' => $method,
        ], 201);
    }

    /**
     * Show single payment method.
     */
    public function show(PaymentMethod $paymentMethod)
    {
        return response()->json(['data' => $paymentMethod]);
    }

    /**
     * Update payment method.
     */
    public function update(Request $request, PaymentMethod $paymentMethod)
    {
        $request->validate([
            'name_ar' => 'sometimes|required|string|max:255',
            'name_en' => 'sometimes|required|string|max:255',
            'description_ar' => 'nullable|string|max:500',
            'description_en' => 'nullable|string|max:500',
            'icon' => 'nullable|image|max:1024',
            'type' => 'sometimes|required|string|max:50',
            'instructions_ar' => 'nullable|string',
            'instructions_en' => 'nullable|string',
            'sort_order' => 'integer|min:0',
            'is_active' => 'boolean',
        ]);

        if ($request->hasFile('icon')) {
            if ($paymentMethod->icon) {
                Storage::disk('public')->delete($paymentMethod->icon);
            }
            $paymentMethod->icon = $request->file('icon')->store('payment-methods', 'public');
        }

        $paymentMethod->update(array_merge(
            $request->except('icon'),
            ['icon' => $paymentMethod->icon]
        ));

        return response()->json([
            'message' => 'تم تحديث طريقة الدفع بنجاح',
            'data' => $paymentMethod,
        ]);
    }

    /**
     * Delete payment method.
     */
    public function destroy(PaymentMethod $paymentMethod)
    {
        if ($paymentMethod->icon) {
            Storage::disk('public')->delete($paymentMethod->icon);
        }
        $paymentMethod->delete();

        return response()->json([
            'message' => 'تم حذف طريقة الدفع بنجاح',
        ]);
    }
}
