<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\OrderStatus;
use App\Models\ProductWeight;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class OrderController extends Controller
{
    /**
     * List all orders (admin).
     */
    public function index(Request $request)
    {
        $query = Order::with(['customer:id,name,phone', 'items']);

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by date range
        if ($request->filled('from')) {
            $query->whereDate('created_at', '>=', $request->from);
        }
        if ($request->filled('to')) {
            $query->whereDate('created_at', '<=', $request->to);
        }

        // Search by order number or customer
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('order_number', 'like', "%{$search}%")
                    ->orWhereHas('customer', function ($cq) use ($search) {
                        $cq->where('name', 'like', "%{$search}%")
                            ->orWhere('phone', 'like', "%{$search}%");
                    });
            });
        }

        $orders = $query->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 20));

        return response()->json($orders);
    }

    /**
     * Show single order.
     */
    public function show(Order $order)
    {
        $order->load([
            'customer',
            'items.product.images',
            'items.productWeight.weightUnit',
            'statuses.user:id,name',
            'coupon',
            'deliveryZone',
            'paymentMethod',
        ]);

        return response()->json(['data' => $order]);
    }

    /**
     * Create order manually (admin).
     */
    public function store(Request $request)
    {
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.product_weight_id' => 'required|exists:product_weights,id',
            'items.*.quantity' => 'required|integer|min:1',
            'delivery_address' => 'required|string|max:500',
            'delivery_city' => 'required|string|max:100',
            'delivery_zone_id' => 'nullable|exists:delivery_zones,id',
            'payment_method_id' => 'nullable|exists:payment_methods,id',
            'discount_amount' => 'nullable|numeric|min:0',
            'delivery_fee' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string|max:1000',
            'status' => 'nullable|string|in:pending,processing,shipped,delivered,cancelled',
        ]);

        return DB::transaction(function () use ($request) {
            $today = now()->format('Ymd');
            $todayCount = Order::whereDate('created_at', now()->toDateString())->count() + 1;
            $orderNumber = 'RN-' . $today . '-' . str_pad($todayCount, 4, '0', STR_PAD_LEFT);

            $subtotal = 0;
            $orderItems = [];

            foreach ($request->items as $itemData) {
                $weight = ProductWeight::with(['product', 'weightUnit'])->findOrFail($itemData['product_weight_id']);
                $itemTotal = $weight->price * $itemData['quantity'];
                $subtotal += $itemTotal;

                $orderItems[] = [
                    'product_id' => $itemData['product_id'],
                    'product_weight_id' => $itemData['product_weight_id'],
                    'product_name' => $weight->product->name_ar,
                    'weight_label' => $weight->weight_value . ' ' . ($weight->weightUnit->abbreviation_ar ?? ''),
                    'unit_price' => $weight->price,
                    'quantity' => $itemData['quantity'],
                    'total_price' => $itemTotal,
                ];
            }

            $discount = $request->get('discount_amount', 0);
            $deliveryFee = $request->get('delivery_fee', 0);
            $total = $subtotal - $discount + $deliveryFee;

            $order = Order::create([
                'order_number' => $orderNumber,
                'customer_id' => $request->customer_id,
                'subtotal' => round($subtotal, 3),
                'discount_amount' => round($discount, 3),
                'delivery_fee' => round($deliveryFee, 3),
                'total' => round($total, 3),
                'status' => $request->get('status', 'pending'),
                'delivery_address' => $request->delivery_address,
                'delivery_city' => $request->delivery_city,
                'delivery_zone_id' => $request->delivery_zone_id,
                'payment_method_id' => $request->payment_method_id,
                'notes' => $request->notes,
            ]);

            foreach ($orderItems as $item) {
                $order->items()->create($item);
            }

            // Log status
            OrderStatus::create([
                'order_id' => $order->id,
                'status' => $order->status,
                'notes' => 'تم إنشاء الطلب بواسطة الإدارة',
                'changed_by' => $request->user()->id,
            ]);

            $order->load(['customer', 'items.product.images']);

            return response()->json([
                'message' => 'تم إنشاء الطلب بنجاح',
                'data' => $order,
            ], 201);
        });
    }

    /**
     * Update order.
     */
    public function update(Request $request, Order $order)
    {
        $request->validate([
            'delivery_address' => 'nullable|string|max:500',
            'delivery_city' => 'nullable|string|max:100',
            'delivery_zone_id' => 'nullable|exists:delivery_zones,id',
            'payment_method_id' => 'nullable|exists:payment_methods,id',
            'discount_amount' => 'nullable|numeric|min:0',
            'delivery_fee' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string|max:1000',
        ]);

        $order->update($request->only([
            'delivery_address', 'delivery_city', 'delivery_zone_id',
            'payment_method_id', 'discount_amount', 'delivery_fee', 'notes',
        ]));

        // Recalculate total
        $subtotal = $order->items->sum('total_price');
        $order->update([
            'subtotal' => round($subtotal, 3),
            'total' => round($subtotal - $order->discount_amount + $order->delivery_fee, 3),
        ]);

        return response()->json([
            'message' => 'تم تحديث الطلب بنجاح',
            'data' => $order->fresh(['customer', 'items']),
        ]);
    }

    /**
     * Delete order.
     */
    public function destroy(Order $order)
    {
        if (in_array($order->status, ['shipped', 'delivered'])) {
            return response()->json([
                'message' => 'لا يمكن حذف طلب تم شحنه أو تسليمه',
            ], 422);
        }

        // Restore stock
        foreach ($order->items as $item) {
            if ($item->productWeight) {
                $item->productWeight->increment('stock', $item->quantity);
            }
            if ($item->product) {
                $item->product->decrement('sales_count', $item->quantity);
            }
        }

        $order->items()->delete();
        $order->statuses()->delete();
        $order->delete();

        return response()->json([
            'message' => 'تم حذف الطلب بنجاح',
        ]);
    }

    /**
     * Update order status.
     */
    public function updateStatus(Request $request, Order $order)
    {
        $request->validate([
            'status' => 'required|string|in:pending,processing,shipped,delivered,cancelled',
            'notes' => 'nullable|string|max:500',
        ]);

        $oldStatus = $order->status;
        $order->update(['status' => $request->status]);

        OrderStatus::create([
            'order_id' => $order->id,
            'status' => $request->status,
            'notes' => $request->notes ?? "تم تغيير الحالة من {$oldStatus} إلى {$request->status}",
            'changed_by' => $request->user()->id,
        ]);

        // If cancelled, restore stock
        if ($request->status === 'cancelled' && $oldStatus !== 'cancelled') {
            foreach ($order->items as $item) {
                if ($item->productWeight) {
                    $item->productWeight->increment('stock', $item->quantity);
                }
                if ($item->product) {
                    $item->product->decrement('sales_count', $item->quantity);
                }
            }
        }

        return response()->json([
            'message' => 'تم تحديث حالة الطلب بنجاح',
            'data' => $order->fresh(['statuses']),
        ]);
    }

    /**
     * Add item to existing order.
     */
    public function addItem(Request $request, Order $order)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'product_weight_id' => 'required|exists:product_weights,id',
            'quantity' => 'required|integer|min:1',
        ]);

        $weight = ProductWeight::with(['product', 'weightUnit'])->findOrFail($request->product_weight_id);

        $itemTotal = $weight->price * $request->quantity;

        $item = $order->items()->create([
            'product_id' => $request->product_id,
            'product_weight_id' => $request->product_weight_id,
            'product_name' => $weight->product->name_ar,
            'weight_label' => $weight->weight_value . ' ' . ($weight->weightUnit->abbreviation_ar ?? ''),
            'unit_price' => $weight->price,
            'quantity' => $request->quantity,
            'total_price' => $itemTotal,
        ]);

        // Recalculate totals
        $subtotal = $order->items()->sum('total_price');
        $order->update([
            'subtotal' => round($subtotal, 3),
            'total' => round($subtotal - $order->discount_amount + $order->delivery_fee, 3),
        ]);

        return response()->json([
            'message' => 'تمت إضافة المنتج إلى الطلب',
            'data' => $item,
        ], 201);
    }

    /**
     * Remove item from order.
     */
    public function removeItem(Order $order, OrderItem $item)
    {
        if ($item->order_id !== $order->id) {
            return response()->json(['message' => 'العنصر لا ينتمي لهذا الطلب'], 422);
        }

        $item->delete();

        // Recalculate totals
        $subtotal = $order->items()->sum('total_price');
        $order->update([
            'subtotal' => round($subtotal, 3),
            'total' => round($subtotal - $order->discount_amount + $order->delivery_fee, 3),
        ]);

        return response()->json([
            'message' => 'تم حذف المنتج من الطلب',
        ]);
    }

    /**
     * Generate invoice data.
     */
    public function invoice(Order $order)
    {
        $order->load([
            'customer',
            'items.product',
            'items.productWeight.weightUnit',
            'deliveryZone',
            'paymentMethod',
        ]);

        return response()->json([
            'data' => [
                'order' => $order,
                'generated_at' => now()->toIso8601String(),
            ],
        ]);
    }

    /**
     * Export orders.
     */
    public function export(Request $request)
    {
        $request->validate([
            'from' => 'nullable|date',
            'to' => 'nullable|date',
            'status' => 'nullable|string',
        ]);

        $query = Order::with(['customer:id,name,phone', 'items']);

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        if ($request->filled('from')) {
            $query->whereDate('created_at', '>=', $request->from);
        }
        if ($request->filled('to')) {
            $query->whereDate('created_at', '<=', $request->to);
        }

        $orders = $query->orderBy('created_at', 'desc')->get();

        $exportData = $orders->map(function ($order) {
            return [
                'order_number' => $order->order_number,
                'customer_name' => $order->customer->name ?? '',
                'customer_phone' => $order->customer->phone ?? '',
                'status' => $order->status,
                'subtotal' => $order->subtotal,
                'discount_amount' => $order->discount_amount,
                'delivery_fee' => $order->delivery_fee,
                'total' => $order->total,
                'delivery_address' => $order->delivery_address,
                'delivery_city' => $order->delivery_city,
                'notes' => $order->notes,
                'items_count' => $order->items->count(),
                'created_at' => $order->created_at->format('Y-m-d H:i:s'),
            ];
        });

        return response()->json(['data' => $exportData]);
    }
}
