<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\GiftCard;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class GiftCardController extends Controller
{
    /**
     * List all gift cards.
     */
    public function index(Request $request)
    {
        $query = GiftCard::query();

        if ($request->filled('search')) {
            $query->where('code', 'like', "%{$request->search}%");
        }

        if ($request->filled('is_active')) {
            $query->where('is_active', $request->boolean('is_active'));
        }

        return response()->json(
            $query->orderBy('created_at', 'desc')
                ->paginate($request->get('per_page', 20))
        );
    }

    /**
     * Create gift card.
     */
    public function store(Request $request)
    {
        $request->validate([
            'code' => 'nullable|string|max:50|unique:gift_cards,code',
            'amount' => 'required|numeric|min:0.001',
            'balance' => 'nullable|numeric|min:0',
            'is_active' => 'boolean',
            'expires_at' => 'nullable|date|after:today',
            'recipient_name' => 'nullable|string|max:255',
            'recipient_phone' => 'nullable|string|max:20',
            'message' => 'nullable|string|max:500',
        ]);

        $code = $request->code ?? strtoupper('GC-' . Str::random(8));

        $giftCard = GiftCard::create([
            'code' => $code,
            'amount' => $request->amount,
            'balance' => $request->get('balance', $request->amount),
            'is_active' => $request->boolean('is_active', true),
            'expires_at' => $request->expires_at,
            'recipient_name' => $request->recipient_name,
            'recipient_phone' => $request->recipient_phone,
            'message' => $request->message,
        ]);

        return response()->json([
            'message' => 'تم إنشاء بطاقة الهدية بنجاح',
            'data' => $giftCard,
        ], 201);
    }

    /**
     * Show single gift card.
     */
    public function show(GiftCard $giftCard)
    {
        return response()->json(['data' => $giftCard]);
    }

    /**
     * Update gift card.
     */
    public function update(Request $request, GiftCard $giftCard)
    {
        $request->validate([
            'code' => 'sometimes|required|string|max:50|unique:gift_cards,code,' . $giftCard->id,
            'amount' => 'sometimes|required|numeric|min:0.001',
            'balance' => 'nullable|numeric|min:0',
            'is_active' => 'boolean',
            'expires_at' => 'nullable|date',
            'recipient_name' => 'nullable|string|max:255',
            'recipient_phone' => 'nullable|string|max:20',
            'message' => 'nullable|string|max:500',
        ]);

        $giftCard->update($request->all());

        return response()->json([
            'message' => 'تم تحديث بطاقة الهدية بنجاح',
            'data' => $giftCard,
        ]);
    }

    /**
     * Delete gift card.
     */
    public function destroy(GiftCard $giftCard)
    {
        $giftCard->delete();

        return response()->json([
            'message' => 'تم حذف بطاقة الهدية بنجاح',
        ]);
    }
}
