<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\Discount;
use Illuminate\Http\Request;

class DiscountController extends Controller
{
    /**
     * List all discounts.
     */
    public function index(Request $request)
    {
        $query = Discount::with(['products:id,name_ar,name_en', 'categories:id,name_ar,name_en']);

        if ($request->filled('is_active')) {
            $query->where('is_active', $request->boolean('is_active'));
        }

        return response()->json(
            $query->orderBy('created_at', 'desc')
                ->paginate($request->get('per_page', 20))
        );
    }

    /**
     * Create discount.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name_ar' => 'required|string|max:255',
            'name_en' => 'required|string|max:255',
            'type' => 'required|in:percentage,fixed',
            'value' => 'required|numeric|min:0',
            'applies_to' => 'required|in:all,products,categories',
            'product_ids' => 'nullable|array',
            'product_ids.*' => 'exists:products,id',
            'category_ids' => 'nullable|array',
            'category_ids.*' => 'exists:categories,id',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'is_active' => 'boolean',
        ]);

        $discount = Discount::create($request->except(['product_ids', 'category_ids']));

        if ($request->has('product_ids')) {
            $discount->products()->sync($request->product_ids);
        }

        if ($request->has('category_ids')) {
            $discount->categories()->sync($request->category_ids);
        }

        $discount->load(['products', 'categories']);

        return response()->json([
            'message' => 'تم إنشاء الخصم بنجاح',
            'data' => $discount,
        ], 201);
    }

    /**
     * Show single discount.
     */
    public function show(Discount $discount)
    {
        $discount->load(['products:id,name_ar,name_en', 'categories:id,name_ar,name_en']);

        return response()->json(['data' => $discount]);
    }

    /**
     * Update discount.
     */
    public function update(Request $request, Discount $discount)
    {
        $request->validate([
            'name_ar' => 'sometimes|required|string|max:255',
            'name_en' => 'sometimes|required|string|max:255',
            'type' => 'sometimes|required|in:percentage,fixed',
            'value' => 'sometimes|required|numeric|min:0',
            'applies_to' => 'sometimes|required|in:all,products,categories',
            'product_ids' => 'nullable|array',
            'product_ids.*' => 'exists:products,id',
            'category_ids' => 'nullable|array',
            'category_ids.*' => 'exists:categories,id',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'is_active' => 'boolean',
        ]);

        $discount->update($request->except(['product_ids', 'category_ids']));

        if ($request->has('product_ids')) {
            $discount->products()->sync($request->product_ids);
        }

        if ($request->has('category_ids')) {
            $discount->categories()->sync($request->category_ids);
        }

        $discount->load(['products', 'categories']);

        return response()->json([
            'message' => 'تم تحديث الخصم بنجاح',
            'data' => $discount,
        ]);
    }

    /**
     * Delete discount.
     */
    public function destroy(Discount $discount)
    {
        $discount->products()->detach();
        $discount->categories()->detach();
        $discount->delete();

        return response()->json([
            'message' => 'تم حذف الخصم بنجاح',
        ]);
    }
}
