<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\DeliveryZone;
use Illuminate\Http\Request;

class DeliveryZoneController extends Controller
{
    /**
     * List all delivery zones.
     */
    public function index()
    {
        $zones = DeliveryZone::orderBy('sort_order', 'asc')
            ->orderBy('name_ar', 'asc')
            ->get();

        return response()->json(['data' => $zones]);
    }

    /**
     * Create delivery zone.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name_ar' => 'required|string|max:255',
            'name_en' => 'required|string|max:255',
            'delivery_fee' => 'required|numeric|min:0',
            'min_order_amount' => 'nullable|numeric|min:0',
            'estimated_days' => 'nullable|integer|min:0',
            'estimated_time_ar' => 'nullable|string|max:100',
            'estimated_time_en' => 'nullable|string|max:100',
            'sort_order' => 'integer|min:0',
            'is_active' => 'boolean',
        ]);

        $zone = DeliveryZone::create([
            'name_ar' => $request->name_ar,
            'name_en' => $request->name_en,
            'delivery_fee' => $request->delivery_fee,
            'min_order_amount' => $request->min_order_amount,
            'estimated_days' => $request->estimated_days,
            'estimated_time_ar' => $request->estimated_time_ar,
            'estimated_time_en' => $request->estimated_time_en,
            'sort_order' => $request->get('sort_order', 0),
            'is_active' => $request->boolean('is_active', true),
        ]);

        return response()->json([
            'message' => 'تم إنشاء منطقة التوصيل بنجاح',
            'data' => $zone,
        ], 201);
    }

    /**
     * Show single delivery zone.
     */
    public function show(DeliveryZone $deliveryZone)
    {
        return response()->json(['data' => $deliveryZone]);
    }

    /**
     * Update delivery zone.
     */
    public function update(Request $request, DeliveryZone $deliveryZone)
    {
        $request->validate([
            'name_ar' => 'sometimes|required|string|max:255',
            'name_en' => 'sometimes|required|string|max:255',
            'delivery_fee' => 'sometimes|required|numeric|min:0',
            'min_order_amount' => 'nullable|numeric|min:0',
            'estimated_days' => 'nullable|integer|min:0',
            'estimated_time_ar' => 'nullable|string|max:100',
            'estimated_time_en' => 'nullable|string|max:100',
            'sort_order' => 'integer|min:0',
            'is_active' => 'boolean',
        ]);

        $deliveryZone->update($request->all());

        return response()->json([
            'message' => 'تم تحديث منطقة التوصيل بنجاح',
            'data' => $deliveryZone,
        ]);
    }

    /**
     * Delete delivery zone.
     */
    public function destroy(DeliveryZone $deliveryZone)
    {
        $deliveryZone->delete();

        return response()->json([
            'message' => 'تم حذف منطقة التوصيل بنجاح',
        ]);
    }
}
