<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\Order;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    /**
     * Overview stats.
     */
    public function index()
    {
        $totalSales = Order::where('status', '!=', 'cancelled')->sum('total');
        $totalOrders = Order::count();
        $totalCustomers = Customer::count();
        $totalProducts = Product::count();

        $pendingOrders = Order::where('status', 'pending')->count();
        $processingOrders = Order::where('status', 'processing')->count();

        // Today's stats
        $todaySales = Order::whereDate('created_at', today())
            ->where('status', '!=', 'cancelled')
            ->sum('total');
        $todayOrders = Order::whereDate('created_at', today())->count();

        // Top products
        $topProducts = Product::orderBy('sales_count', 'desc')
            ->limit(10)
            ->get(['id', 'name_ar', 'name_en', 'sales_count', 'base_price']);

        // Recent orders
        $recentOrders = Order::with(['customer:id,name,phone'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        // Monthly sales (last 12 months)
        $monthlySales = Order::where('status', '!=', 'cancelled')
            ->where('created_at', '>=', now()->subMonths(12))
            ->select(
                DB::raw('YEAR(created_at) as year'),
                DB::raw('MONTH(created_at) as month'),
                DB::raw('SUM(total) as total'),
                DB::raw('COUNT(*) as orders_count')
            )
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get();

        return response()->json([
            'data' => [
                'total_sales' => round($totalSales, 3),
                'total_orders' => $totalOrders,
                'total_customers' => $totalCustomers,
                'total_products' => $totalProducts,
                'pending_orders' => $pendingOrders,
                'processing_orders' => $processingOrders,
                'today_sales' => round($todaySales, 3),
                'today_orders' => $todayOrders,
                'top_products' => $topProducts,
                'recent_orders' => $recentOrders,
                'monthly_sales' => $monthlySales,
            ],
        ]);
    }

    /**
     * Detailed stats with date ranges.
     */
    public function stats(Request $request)
    {
        $request->validate([
            'from' => 'nullable|date',
            'to' => 'nullable|date|after_or_equal:from',
        ]);

        $from = $request->get('from', now()->subDays(30)->toDateString());
        $to = $request->get('to', now()->toDateString());

        $salesQuery = Order::where('status', '!=', 'cancelled')
            ->whereBetween('created_at', [$from, $to . ' 23:59:59']);

        $totalSales = (clone $salesQuery)->sum('total');
        $totalOrders = (clone $salesQuery)->count();
        $avgOrderValue = $totalOrders > 0 ? $totalSales / $totalOrders : 0;

        $newCustomers = Customer::whereBetween('created_at', [$from, $to . ' 23:59:59'])->count();

        // Daily breakdown
        $dailySales = (clone $salesQuery)
            ->select(
                DB::raw('DATE(created_at) as date'),
                DB::raw('SUM(total) as total'),
                DB::raw('COUNT(*) as orders_count')
            )
            ->groupBy('date')
            ->orderBy('date', 'asc')
            ->get();

        // Orders by status
        $ordersByStatus = Order::whereBetween('created_at', [$from, $to . ' 23:59:59'])
            ->select('status', DB::raw('COUNT(*) as count'))
            ->groupBy('status')
            ->get()
            ->pluck('count', 'status');

        // Top categories
        $topCategories = DB::table('order_items')
            ->join('products', 'order_items.product_id', '=', 'products.id')
            ->join('categories', 'products.category_id', '=', 'categories.id')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->where('orders.status', '!=', 'cancelled')
            ->whereBetween('orders.created_at', [$from, $to . ' 23:59:59'])
            ->select(
                'categories.id',
                'categories.name_ar',
                'categories.name_en',
                DB::raw('SUM(order_items.total) as total_sales'),
                DB::raw('SUM(order_items.quantity) as total_quantity')
            )
            ->groupBy('categories.id', 'categories.name_ar', 'categories.name_en')
            ->orderBy('total_sales', 'desc')
            ->limit(10)
            ->get();

        return response()->json([
            'data' => [
                'period' => ['from' => $from, 'to' => $to],
                'total_sales' => round($totalSales, 3),
                'total_orders' => $totalOrders,
                'avg_order_value' => round($avgOrderValue, 3),
                'new_customers' => $newCustomers,
                'daily_sales' => $dailySales,
                'orders_by_status' => $ordersByStatus,
                'top_categories' => $topCategories,
            ],
        ]);
    }
}
