<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\Currency;
use Illuminate\Http\Request;

class CurrencyController extends Controller
{
    /**
     * List all currencies.
     */
    public function index()
    {
        $currencies = Currency::orderBy('is_default', 'desc')
            ->orderBy('name_ar', 'asc')
            ->get();

        return response()->json(['data' => $currencies]);
    }

    /**
     * Create currency.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name_ar' => 'required|string|max:100',
            'name_en' => 'required|string|max:100',
            'code' => 'required|string|max:10|unique:currencies,code',
            'symbol' => 'required|string|max:10',
            'exchange_rate' => 'required|numeric|min:0.0001',
            'is_default' => 'boolean',
            'is_active' => 'boolean',
            'decimal_places' => 'integer|min:0|max:4',
        ]);

        // If setting as default, unset current default
        if ($request->boolean('is_default')) {
            Currency::where('is_default', true)->update(['is_default' => false]);
        }

        $currency = Currency::create([
            'name_ar' => $request->name_ar,
            'name_en' => $request->name_en,
            'code' => strtoupper($request->code),
            'symbol' => $request->symbol,
            'exchange_rate' => $request->exchange_rate,
            'is_default' => $request->boolean('is_default', false),
            'is_active' => $request->boolean('is_active', true),
            'decimal_places' => $request->get('decimal_places', 3),
        ]);

        return response()->json([
            'message' => 'تم إنشاء العملة بنجاح',
            'data' => $currency,
        ], 201);
    }

    /**
     * Show single currency.
     */
    public function show(Currency $currency)
    {
        return response()->json(['data' => $currency]);
    }

    /**
     * Update currency.
     */
    public function update(Request $request, Currency $currency)
    {
        $request->validate([
            'name_ar' => 'sometimes|required|string|max:100',
            'name_en' => 'sometimes|required|string|max:100',
            'code' => 'sometimes|required|string|max:10|unique:currencies,code,' . $currency->id,
            'symbol' => 'sometimes|required|string|max:10',
            'exchange_rate' => 'sometimes|required|numeric|min:0.0001',
            'is_default' => 'boolean',
            'is_active' => 'boolean',
            'decimal_places' => 'integer|min:0|max:4',
        ]);

        if ($request->boolean('is_default') && !$currency->is_default) {
            Currency::where('is_default', true)->update(['is_default' => false]);
        }

        $currency->update($request->all());

        return response()->json([
            'message' => 'تم تحديث العملة بنجاح',
            'data' => $currency,
        ]);
    }

    /**
     * Delete currency.
     */
    public function destroy(Currency $currency)
    {
        if ($currency->is_default) {
            return response()->json([
                'message' => 'لا يمكن حذف العملة الافتراضية',
            ], 422);
        }

        $currency->delete();

        return response()->json([
            'message' => 'تم حذف العملة بنجاح',
        ]);
    }
}
