<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\Coupon;
use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class CouponController extends Controller
{
    /**
     * List all coupons.
     */
    public function index(Request $request)
    {
        $query = Coupon::query();

        if ($request->filled('search')) {
            $query->where('code', 'like', "%{$request->search}%");
        }

        if ($request->filled('is_active')) {
            $query->where('is_active', $request->boolean('is_active'));
        }

        return response()->json(
            $query->orderBy('created_at', 'desc')
                ->paginate($request->get('per_page', 20))
        );
    }

    /**
     * Create coupon.
     */
    public function store(Request $request)
    {
        $request->validate([
            'code' => 'required|string|max:50|unique:coupons,code',
            'type' => 'required|in:percentage,fixed',
            'value' => 'required|numeric|min:0',
            'max_discount' => 'nullable|numeric|min:0',
            'min_order_amount' => 'nullable|numeric|min:0',
            'max_uses' => 'nullable|integer|min:1',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'is_active' => 'boolean',
            'description' => 'nullable|string|max:500',
        ]);

        $coupon = Coupon::create($request->all());

        return response()->json([
            'message' => 'تم إنشاء الكوبون بنجاح',
            'data' => $coupon,
        ], 201);
    }

    /**
     * Show single coupon.
     */
    public function show(Coupon $coupon)
    {
        return response()->json(['data' => $coupon]);
    }

    /**
     * Update coupon.
     */
    public function update(Request $request, Coupon $coupon)
    {
        $request->validate([
            'code' => 'sometimes|required|string|max:50|unique:coupons,code,' . $coupon->id,
            'type' => 'sometimes|required|in:percentage,fixed',
            'value' => 'sometimes|required|numeric|min:0',
            'max_discount' => 'nullable|numeric|min:0',
            'min_order_amount' => 'nullable|numeric|min:0',
            'max_uses' => 'nullable|integer|min:1',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'is_active' => 'boolean',
            'description' => 'nullable|string|max:500',
        ]);

        $coupon->update($request->all());

        return response()->json([
            'message' => 'تم تحديث الكوبون بنجاح',
            'data' => $coupon,
        ]);
    }

    /**
     * Delete coupon.
     */
    public function destroy(Coupon $coupon)
    {
        $coupon->delete();

        return response()->json([
            'message' => 'تم حذف الكوبون بنجاح',
        ]);
    }

    /**
     * Coupon usage stats.
     */
    public function stats(Coupon $coupon)
    {
        $ordersUsing = Order::where('coupon_id', $coupon->id)
            ->where('status', '!=', 'cancelled');

        $totalOrdersUsing = (clone $ordersUsing)->count();
        $totalDiscountGiven = (clone $ordersUsing)->sum('discount');
        $totalRevenue = (clone $ordersUsing)->sum('total');

        $recentOrders = (clone $ordersUsing)
            ->with(['customer:id,name,phone'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return response()->json([
            'data' => [
                'used_count' => $coupon->used_count,
                'max_uses' => $coupon->max_uses,
                'total_orders_using' => $totalOrdersUsing,
                'total_discount_given' => round($totalDiscountGiven, 3),
                'total_revenue' => round($totalRevenue, 3),
                'recent_orders' => $recentOrders,
            ],
        ]);
    }
}
