<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\ChatConversation;
use App\Models\ChatMessage;
use App\Models\Setting;
use Illuminate\Http\Request;

class ChatController extends Controller
{
    /**
     * List all conversations.
     */
    public function conversations(Request $request)
    {
        $query = ChatConversation::with([
            'customer:id,name,phone',
            'lastMessage',
        ])->withCount(['messages as unread_count' => function ($q) {
            $q->where('is_read', false)->where('sender_type', 'customer');
        }]);

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('subject', 'like', "%{$search}%")
                    ->orWhereHas('customer', function ($cq) use ($search) {
                        $cq->where('name', 'like', "%{$search}%")
                            ->orWhere('phone', 'like', "%{$search}%");
                    });
            });
        }

        $conversations = $query->orderBy('updated_at', 'desc')
            ->paginate($request->get('per_page', 20));

        return response()->json($conversations);
    }

    /**
     * Show single conversation with messages.
     */
    public function show(ChatConversation $conversation)
    {
        $conversation->load(['customer', 'messages' => function ($q) {
            $q->orderBy('created_at', 'asc');
        }]);

        // Mark customer messages as read
        $conversation->messages()
            ->where('sender_type', 'customer')
            ->where('is_read', false)
            ->update(['is_read' => true]);

        return response()->json(['data' => $conversation]);
    }

    /**
     * Send a message as admin.
     */
    public function sendMessage(Request $request, ChatConversation $conversation)
    {
        $request->validate([
            'message' => 'required|string|max:2000',
            'attachment' => 'nullable|file|max:5120',
        ]);

        $attachmentPath = null;
        if ($request->hasFile('attachment')) {
            $attachmentPath = $request->file('attachment')->store('chat-attachments', 'public');
        }

        $message = ChatMessage::create([
            'chat_conversation_id' => $conversation->id,
            'sender_type' => 'admin',
            'sender_id' => $request->user()->id,
            'message' => $request->message,
            'attachment' => $attachmentPath,
        ]);

        $conversation->touch();

        return response()->json([
            'message' => 'تم إرسال الرسالة',
            'data' => $message,
        ], 201);
    }

    /**
     * Get chat settings.
     */
    public function settings()
    {
        $settings = Setting::where('key', 'like', 'chat_%')
            ->get()
            ->pluck('value', 'key');

        return response()->json(['data' => $settings]);
    }

    /**
     * Update chat settings.
     */
    public function updateSettings(Request $request)
    {
        $request->validate([
            'chat_enabled' => 'nullable|boolean',
            'chat_welcome_message_ar' => 'nullable|string|max:500',
            'chat_welcome_message_en' => 'nullable|string|max:500',
            'chat_auto_reply_ar' => 'nullable|string|max:500',
            'chat_auto_reply_en' => 'nullable|string|max:500',
            'chat_working_hours_start' => 'nullable|string|max:10',
            'chat_working_hours_end' => 'nullable|string|max:10',
        ]);

        foreach ($request->all() as $key => $value) {
            if (str_starts_with($key, 'chat_')) {
                Setting::updateOrCreate(
                    ['key' => $key],
                    ['value' => (string) $value, 'group' => 'chat', 'is_public' => false]
                );
            }
        }

        return response()->json([
            'message' => 'تم تحديث إعدادات الدردشة بنجاح',
        ]);
    }
}
