<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\Banner;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class BannerController extends Controller
{
    /**
     * List all banners.
     */
    public function index()
    {
        $banners = Banner::orderBy('sort_order', 'asc')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json(['data' => $banners]);
    }

    /**
     * Create banner.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title_ar' => 'nullable|string|max:255',
            'title_en' => 'nullable|string|max:255',
            'subtitle_ar' => 'nullable|string|max:500',
            'subtitle_en' => 'nullable|string|max:500',
            'image' => 'required|image|max:5120',
            'image_mobile' => 'nullable|image|max:5120',
            'link' => 'nullable|string|max:500',
            'link_type' => 'nullable|in:internal,external,product,category',
            'sort_order' => 'integer|min:0',
            'is_active' => 'boolean',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        $imagePath = $request->file('image')->store('banners', 'public');
        $mobileImagePath = null;
        if ($request->hasFile('image_mobile')) {
            $mobileImagePath = $request->file('image_mobile')->store('banners', 'public');
        }

        $banner = Banner::create(array_merge(
            $request->except(['image', 'image_mobile']),
            [
                'image' => $imagePath,
                'image_mobile' => $mobileImagePath,
                'is_active' => $request->boolean('is_active', true),
                'sort_order' => $request->get('sort_order', 0),
            ]
        ));

        return response()->json([
            'message' => 'تم إنشاء البانر بنجاح',
            'data' => $banner,
        ], 201);
    }

    /**
     * Show single banner.
     */
    public function show(Banner $banner)
    {
        return response()->json(['data' => $banner]);
    }

    /**
     * Update banner.
     */
    public function update(Request $request, Banner $banner)
    {
        $request->validate([
            'title_ar' => 'nullable|string|max:255',
            'title_en' => 'nullable|string|max:255',
            'subtitle_ar' => 'nullable|string|max:500',
            'subtitle_en' => 'nullable|string|max:500',
            'image' => 'nullable|image|max:5120',
            'image_mobile' => 'nullable|image|max:5120',
            'link' => 'nullable|string|max:500',
            'link_type' => 'nullable|in:internal,external,product,category',
            'sort_order' => 'integer|min:0',
            'is_active' => 'boolean',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        if ($request->hasFile('image')) {
            Storage::disk('public')->delete($banner->image);
            $banner->image = $request->file('image')->store('banners', 'public');
        }

        if ($request->hasFile('image_mobile')) {
            if ($banner->image_mobile) {
                Storage::disk('public')->delete($banner->image_mobile);
            }
            $banner->image_mobile = $request->file('image_mobile')->store('banners', 'public');
        }

        $banner->update(array_merge(
            $request->except(['image', 'image_mobile']),
            ['image' => $banner->image, 'image_mobile' => $banner->image_mobile]
        ));

        return response()->json([
            'message' => 'تم تحديث البانر بنجاح',
            'data' => $banner,
        ]);
    }

    /**
     * Delete banner.
     */
    public function destroy(Banner $banner)
    {
        Storage::disk('public')->delete($banner->image);
        if ($banner->image_mobile) {
            Storage::disk('public')->delete($banner->image_mobile);
        }
        $banner->delete();

        return response()->json([
            'message' => 'تم حذف البانر بنجاح',
        ]);
    }

    /**
     * Reorder banners.
     */
    public function reorder(Request $request)
    {
        $request->validate([
            'banners' => 'required|array',
            'banners.*.id' => 'required|exists:banners,id',
            'banners.*.sort_order' => 'required|integer|min:0',
        ]);

        foreach ($request->banners as $item) {
            Banner::where('id', $item['id'])->update(['sort_order' => $item['sort_order']]);
        }

        return response()->json([
            'message' => 'تم إعادة ترتيب البانرات بنجاح',
        ]);
    }
}
